/**
 * \file: grl_font.c
 */
/*PRQA: QAC Message 288: invalid chars are always in file header */
/* \version: $Id: grl_font.c,v 1.30 2011/06/16 09:14:24 efriedrich Exp $
 *
 * \component:  helper functions for svg_font library
 *
 * \author:
 *
 * \copyright: (c) 2003 - 2006 ADIT Corporation
 *
 * \history
 *
 ***********************************************************************/

#include "grl_font.h"
#include "svg_font_err.h"
#include "svg_error.h"
#include "grl_configuration.h"
#include "grl_os_abstraction.h"

#include <fcntl.h>
#include <sys/stat.h>
#include <semaphore.h>
#include <sys/mman.h>
#include <unistd.h>

#include <stdlib.h>
#include <string.h>



/****************************** Global Params ********************************/

#define FLASH_FONT          TRUE
#define FILE_FONT           FALSE

/* to select the current module info list */
static   grl_module_info_list    *gp_module_info_list    = NULL;



/* *************************** Local Functions *******************************/

static   SVGBoolean  local_buildup_flash_font_database( void );
static   void        local_buildup_file_font_database( void );
static   SVGBoolean  local_fill_struct(SVGUint32 *p_file_list,
                                      SVGUint32 num_file_names,
                                      grl_module_info* p_module_info,
                                      SVGBoolean is_flash_font);
static   SVGUint32         local_num_of_flash_font(const SVGInt8 *p_suffix_list);
static   SVGBoolean  local_is_valid_suffix(grl_flash_font flash_font,
                                          const SVGInt8 *p_suffix_list);
static   SVGChar     *local_get_flash_font_file(const grl_flash_font* flash_font,
                                              SVGChar *name);
static   SVGError          local_build_file_font_list(SVGUint32       *p_nameList,
                                               SVGUint32      *FileCount,
                                               const SVGInt8 *p_suffix_list);

extern SVGError grl_font_scan_dir( SVGUint32       *p_nameList,
                                   SVGUint32       *p_fileCount,
                                   const SVGChar   *p_suffix_list ,
                                   SVGChar*        svgpathname,
                                   SVGUint32*      p_listCount );
static void update_filepath(void);

static SVGBoolean check_shmemory(SVGChar *shm_name , SVGUint32 *p_shm_size ,
		SVGUint32 *p_file_list,SVGFont** p_font_data,SVGUint32*  p_font_max ,SVGUint32 total_files);


static SVGBoolean create_shmemory(SVGChar *shm_name , SVGUint32 *p_shm_size ,
				SVGFont** p_font_data , SVGUint32*  p_font_max , SVGUint32 *p_file_list);

typedef struct
	{
		SVGFont             flash_data;
		SVGUint32           num_flash_files;
	}shm_struct_flash;

typedef struct
	{
		SVGFont             font_data;
		SVGUint32           num_font_files;
	}shm_struct_font;



static SVGChar     svgpath_fontfiles[256]  = {0};
SVGChar*    	   parsed_file_buffer      = 0;
SVGUint32 	       parsed_file_buffer_size = 0;


sem_t* GRL_FONT_LOCK_ID;


SVGBoolean GRL_buildup_font_database( void )
{
    SVGBoolean      ret     = SVG_TRUE;

    update_filepath();
    strcat(svgpath_fontfiles,"parsed_fontfile.bin");
    /* read flash fonts */
    ret = local_buildup_flash_font_database();
    if (ret == SVG_TRUE)
    {
        /* read file fonts */
        local_buildup_file_font_database();

        /* set list to begin */
        gp_module_info_list = NULL;
    }

    return ret;
}

/**
 * @fn update_filepath
 *
 * updating the path from where to open parsed font file
 *
 * @return none
 */
static void update_filepath(void)
{


	SVGInt32          a_length       = 0;
	SVGUint32         svg_maxpath    = 0;
	SVGChar           devconf_buf[256] = {0};
	SVGChar           *tmp           = NULL;

	memset(&svgpath_fontfiles[0],0, sizeof(svgpath_fontfiles));
	a_length = GRL_get_int_config((SVGChar*)"SVGPATHCOUNT", (SVGInt32*)&svg_maxpath, 0);
	if( a_length > 0)
	{
		GRL_get_int_config((SVGChar*)"SVGPATHCOUNT", (SVGInt32*)&svg_maxpath, (SVGUint8)a_length);
		/* are the pathnames available? */
		a_length = GRL_get_string_config((SVGChar*)"SVGPATHNAMES", (SVGUint8*)&devconf_buf[0], 0);
		if((a_length > 0) && (256 > a_length))
		{
			/* read out the pathnames and use them */
			GRL_get_string_config((SVGChar*)"SVGPATHNAMES", (SVGUint8*)&devconf_buf[0], (SVGUint8)a_length);
			if(1 == svg_maxpath)
			{
				memcpy(&svgpath_fontfiles[0],(SVGUint8*)&devconf_buf[0],(SVGUint32)a_length);
			}
			else
			{
				tmp = strtok((SVGChar*)&devconf_buf,",");
				if (NULL != tmp)
				{
					strncpy(&svgpath_fontfiles[0] ,tmp ,256);
				}
			}

		}
		else
		{
			/* in case of any unavailable or invalid value
				   just use internal paths for parsing */
				svg_maxpath = SVG_MAXPATH;
		}
	}
}

/* PRQA: 310: all cast error for string literals off */
/*PRQA S 310 L1 */

/* PRQA: 3673: all not mofified params in function to 'const' off */
/*PRQA S 3673 L2 */

SVGFont* GRL_get_font_id( SVGFontContext* ctx, SVGFont* p_font )
{
    SVGFont            *p_font_buf    = NULL;
    SVGFont            *p_font_ret    = NULL;
    grl_module_info    *p_module_info = ctx->p_module_info;
    SVGUint32                 i             = 0;

    if (p_font != NULL)
    {
        /* searching in flashfonts */
        for ( i = 0; i < p_module_info->svgFlashFontsMax; i++ )
        {
            p_font_buf = &p_module_info->p_svgFlashFonts[i];
            if ( strcmp( (const SVGChar*)p_font_buf->fontName, (const SVGChar*)p_font->fontName ) == 0 )
            {
                p_font_ret = p_font_buf;
            }
        }

        if (p_font_ret == NULL)
        {
            /* searching in filefonts */
            for ( i = 0; i < p_module_info->svgFileFontsMax; i++ )
            {
                p_font_buf = &p_module_info->p_svgFileFonts[i];
                if (strcmp((const SVGChar*)p_font_buf->fontName, (const SVGChar*)p_font->fontName) == 0)
                {
                    p_font_ret = p_font_buf;
                }
            }
        }
    }

    if (p_font_ret == NULL)
    {
        p_font_ret = p_module_info->p_svgFlashFonts;
        ctx->error = SVG_INVALID_VALUE;
    }

    return p_font_ret;
}


SVGBoolean  GRL_get_next_module_info( grl_module_info** pp_module_info )
{
    SVGBoolean      ret = SVG_TRUE;

    if (gp_module_info_list == NULL)
    {
        gp_module_info_list = gp_SVG_beginModuleInfoList;
    }
    else
    {
        gp_module_info_list = gp_module_info_list->next;
    }

    /* if pointer of info list on end -> no info */
    if (gp_module_info_list != NULL)
    {
        *pp_module_info = gp_module_info_list->p_module_info;
    }
    else
    {
        ret = SVG_FALSE;
    }

    return ret;
}

/**
 * @fn local_buildup_flash_font_database
 *
 * buildup the flash font database
 *
 * @return TRUE or FALSE
 */
static SVGBoolean local_buildup_flash_font_database( void )
{
    SVGBoolean          ret             = SVG_TRUE;
    SVGUint32                 i               = 0;
    grl_module_info    *p_module_info   = NULL;
    SVGBoolean          IsValidSuffix   = SVG_TRUE;
    SVGInt8                 *p_suffix_list   = NULL;
    SVGUint32                 list_size       = 0;
    SVGChar            *p_file_name     = NULL;
    SVGUint32                 *p_file_list     = NULL;
    SVGUint32                 num_files       = 0;

    /* get for every module the flash font informations */
    while(GRL_get_next_module_info( &p_module_info ) == SVG_TRUE)
    {
        /* get count and suffixes off supported flash fonts for this module */
        p_module_info->svgFlashFontsMax = 0;
        p_module_info->p_module->p_grl_fp_table->
            get_font_suffix( &p_suffix_list, &list_size );
        num_files = local_num_of_flash_font(p_suffix_list);

        if (0 != num_files)
        {
            /* allocate pointer buffer for filenames */
            p_file_list  = (SVGUint32*)GRL_malloc_1D_resource(sizeof(SVGUint32) * num_files);
            if (NULL != p_file_list)
            {
                num_files = 0;
                for (i = 0; i < GRL_FLASH_FONT_COUNT; i++)
                {
                    IsValidSuffix =
                        local_is_valid_suffix(flash_fonts[i], p_suffix_list);
                    if (IsValidSuffix == SVG_TRUE)
                    {
                        /* allocate space for name */
                        p_file_name = (SVGChar*)GRL_malloc_1D_resource(FF_NAME_LENGTH);
                        if (NULL != p_file_name)
                        {
                            local_get_flash_font_file(&flash_fonts[i],
                                                      p_file_name);
/* PRQA: 306: pointers are checked */
/*PRQA S 306 L1 */
                            p_file_list[num_files] = (uintptr_t)p_file_name;
                            num_files++;
/*PRQA L L1 */
                        }
                    }
                }

                /*
                 * Create in module the font information list
                 *
                 * Note: here also clears the name space
                 */
                ret = local_fill_struct(p_file_list , num_files,
                                        p_module_info, FLASH_FONT);

                /* delete pointer buffer */
                GRL_free_1D_resource((void*)p_file_list);
            }
        }
        else
        {
            ret = SVG_FALSE;
        }

    }

    return ret;
}


/**
 * @fn local_buildup_file_font_database
 *
 * buildup the file font database
 *
 * @return none
 */
static void local_buildup_file_font_database( void )
{
    SVGError            err             = SVG_NO_ERROR;
    grl_module_info    *p_module_info   = NULL;
    SVGInt8                 *p_suffix_list   = NULL;
    SVGUint32                 list_size      = 0;
    SVGUint32                 *p_file_list   = NULL;
    SVGUint32                 num_files      = 0;

    /* get for every module the file font informations */
    while(GRL_get_next_module_info( &p_module_info ) == SVG_TRUE)
    {
        /* get count and suffixes off supported file fonts for this module */
        p_module_info->svgFileFontsMax  = 0;
        p_module_info->p_module->p_grl_fp_table->
            get_font_suffix(&p_suffix_list, &list_size);

        /* get count of available files */
        err = local_build_file_font_list(NULL, &num_files, p_suffix_list);

        if (err == SVG_NO_ERROR)
        {
            if ( num_files != 0 )
            {
                /* allocate pointer buffer for filenames */
                p_file_list = (SVGUint32*)GRL_malloc_1D_resource(sizeof(SVGUint32) * num_files);
                if (NULL != p_file_list)
                {
                    err = local_build_file_font_list( p_file_list ,
                                                    NULL,
                                                    p_suffix_list );
                    /*
                     * Create in module the font information list
                     *
                     * Note: here also clears the name space
                     */
                    local_fill_struct(p_file_list , num_files,
                                      p_module_info, FILE_FONT);

                    /* delete pointer buffer */
                    GRL_free_1D_resource((void*)p_file_list);
                }
            }
        }
    }
}


/**
 * @fn check_shmemory
 *
 * check for wheather shared memory or parsed font file are present
 *
 * @param shm_name            : shared memory name
 * @param p_file_list         : list of file names
 * @param p_shm_size          : shared memory size
 * @param @param p_font_data  : updated the struct if parsed font file is present
 * @param p_font_max          : updating the total num of faces per file from parsed font file
 *
 * @return TRUE or FALSE
 *
 */

static SVGBoolean check_shmemory(SVGChar *shm_name , SVGUint32 *p_shm_size ,
		SVGUint32 *p_file_list ,SVGFont** p_font_data, SVGUint32*  p_font_max ,SVGUint32 total_files)
{

	SVGInt32    fd  				= 1;
	struct stat shm_stat 			;
	FILE *      fq      			= NULL ;
	SVGBoolean  ret           		= SVG_TRUE;
	SVGUint32   i 					= 0;
	SVGUint32   font_max2       	= 0;
	SVGUint32	k				  	= 0;
	SVGUint32   found         	  	= 0;
	SVGUint32   file_font_offset 	= 0;
	SVGUint32   flash_font_offset 	= 0;
	SVGUint32   num_of_file_fonts 	= 0;
	SVGFont*    p_tmp               = NULL;
	SVGChar     shared_path_per[256]  = "/dev/shm/";
	SVGChar		shm_name_per[256]	= {0};
	SVGChar *   p_string		= NULL;


	fd = shm_open( shm_name, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR );
	if ( fd <= -1)
	{
	    SVG_FNT_E("SHM_OPEN_FAILED IN CHECK_SHMEMORY");
		ret = SVG_FALSE;
	}
	else
	{
		// to remove slash at the beginning of the name
		p_string		= &shm_name[1] ;
		strcpy((char *)shm_name_per ,(char *)shared_path_per);
		strcat((char *)shm_name_per, p_string );
		if(chmod(shm_name_per, S_IRUSR | S_IWUSR | S_IROTH | S_IWOTH )!= 0)
		{
		    SVG_FNT_E("SHM_CHMOD_FAILED IN CHECK_SHMEMORY");
		}

		/* get state */
		if (fstat(fd, &shm_stat ) != 0)
		{
		    SVG_FNT_E("SHM_STATE_FAILED IN CHECK_SHMEMORY");
			ret = SVG_FALSE;
		}
		else
		{
			if (shm_stat.st_size == 0)
			{

				if(parsed_file_buffer == NULL)
				{
					fq=fopen(svgpath_fontfiles,"r");
					if(fq!=NULL)
					{
						fseek(fq , 0 , SEEK_END );
						parsed_file_buffer_size = ftell(fq);
						fseek(fq , 0 , SEEK_SET );
						parsed_file_buffer = (SVGChar *)GRL_malloc_1D_resource( parsed_file_buffer_size );
						fread( parsed_file_buffer , 1 , parsed_file_buffer_size,fq );
					}
					else
					{
						SVG_FNT_E(
							"OPENING_PARSED_FILE_FAILED IN CHECK_SHMEMORY");
						ret         = SVG_FALSE ;
					}
				}

				if(ret == SVG_TRUE)
				{

					if(0 == strcmp(shm_name,"/svg_flashfonts"))
					{

						p_font_max[0] = parsed_file_buffer[PARSED_FLASHFONTS];
						p_shm_size[0] = (sizeof(SVGFont) * p_font_max[0] );

						*p_font_data = (SVGFont*)GRL_malloc_1D_resource( p_shm_size[0] );
						flash_font_offset = PARSED_FILEFONTS ;

						for(i=0; i < p_font_max[0]; i++)
						{
							memcpy(&p_font_data[0][i] ,
									 &parsed_file_buffer[ flash_font_offset] , sizeof(SVGFont) );

							flash_font_offset += ( sizeof(SVGFont)) ;
						}
					}
					else
					{
						font_max2 = parsed_file_buffer[PARSED_FLASHFONTS];
						p_font_max[0] = parsed_file_buffer[ PARSED_FILEFONTS + ( font_max2 * sizeof(SVGFont) )];
						p_shm_size[0] = (sizeof(SVGFont) * p_font_max[0] );

						*p_font_data = (SVGFont*)GRL_malloc_1D_resource( p_shm_size[0] );

						/*file font data starts at the below offset in the parsed file*/
						file_font_offset = PARSED_FILEFONTS + 1 + ( sizeof(SVGFont)*font_max2 );
						num_of_file_fonts = 0;

						for(i=0; i < p_font_max[0]; i++)
						{

							p_tmp = (SVGFont* ) ((void *)&parsed_file_buffer[file_font_offset]);


							/*Assign correct path name to font data by check from the target
							file list because the path names between host and target differ*/
							found = 0;
							for(k=0; ((k < total_files ) && (found == 0)); k++)
							{
								if(0 != strstr( (const SVGChar*)(uintptr_t)p_file_list[k],
												(const SVGChar*)p_tmp->filename))
								{
									memcpy(&p_font_data[0][num_of_file_fonts],
											&parsed_file_buffer[file_font_offset],
												(sizeof(SVGFont)) );

									strcpy((char *)&p_font_data[0][num_of_file_fonts].filename ,
														(char *)(uintptr_t)p_file_list[k] );

									num_of_file_fonts++;
									found =1;
								}
							}
							/* moving the offset by sizeof struct */
							file_font_offset += (sizeof(SVGFont));

						}

						p_font_max[0] = num_of_file_fonts ;
						GRL_free_1D_resource((void*)parsed_file_buffer);
					}
				}
			}
			else
			{
				p_shm_size[0]   = shm_stat.st_size ;
				ret             = SVG_TRUE ;
			}
		}
		close(fd);
	}
	return ret;
}


/**
 * @fn create_shmemory
 *
 * check for wheather shared memory or parsed font file are present
 *
 * @param shm_name            : shared memory name
 * @param p_file_list         : list of file names
 * @param p_shm_size          : shared memory size
 * @param p_font_max          : updating the total num of faces per file from shared memory
 * @param p_font_data         : updated the struct from shared memory
 *
 * @return return TRUE or FALSE
 *
 */
static SVGBoolean create_shmemory(SVGChar *shm_name , SVGUint32 *p_shm_size ,SVGFont** p_font_data , SVGUint32*  p_font_max ,SVGUint32 *p_file_list)
{

	SVGInt32	grl_error  		= 0;
	SVGInt32    fd 				= 1;
	struct stat shm_stat 		;
	SVGBoolean  shm_init   		= SVG_FALSE;
	SVGBoolean  mmap_required   = SVG_FALSE;
	void *      mmap_struct 	= NULL;
	SVGUint32   i 				= 0;
	SVGBoolean  ret             = SVG_TRUE;
	SVGChar     shared_path_per[256]  = "/dev/shm/";
	SVGChar		shm_name_per[256]	= {0};
	SVGChar *   p_string		= NULL;

	shm_struct_flash 	*gp_svgfont_flashshared = NULL;
	shm_struct_font	 	*gp_svgfont_fontshared = NULL;



	GRL_FONT_LOCK_ID = sem_open("/svg_font_sem", O_CREAT, 0777, 1);
	if( GRL_FONT_LOCK_ID == SEM_FAILED || GRL_FONT_LOCK_ID == NULL )
	{
	    SVG_FNT_E("SEM_OPEN_FAILED IN CREATE_SHMEMORY");
		ret = SVG_FALSE;
	}
	else
	{
		if(chmod("/dev/shm/sem.svg_font_sem", S_IRWXU | S_IRWXG |S_IRWXO)!= 0)
		{
		    SVG_FNT_E("SEM_CHMOD_FAILED IN CREATE_SHMEMORY");
		}
		grl_error = sem_wait(GRL_FONT_LOCK_ID);
		if (0 == grl_error)
		{

			fd = shm_open( shm_name, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR );

			if ( fd <= -1 )
			{
			    SVG_FNT_E("SHM_OPEN_FAILED IN CREATE_SHMEMORY");
				ret = SVG_FALSE;
			}
			else
			{
				// to remove slash at the beginning of the name
				p_string		= &shm_name[1] ;
				strcpy((char *)shm_name_per ,(char *)shared_path_per);
				strcat((char *)shm_name_per, p_string);
				if(chmod(shm_name_per, S_IRUSR | S_IWUSR | S_IROTH | S_IWOTH )!= 0)
				{
				    SVG_FNT_E("SHM_CHMOD_FAILED IN CREATE_SHMEMORY");
				}
				/* get state */
				if ( fstat(fd, &shm_stat) != 0)
				{
				    SVG_FNT_E("SHM_STATE_FAILED IN CREATE_SHMEMORY");
					ret = SVG_FALSE;
				}
				else
				{
					p_shm_size[0] = (p_shm_size[0] + (sysconf(_SC_PAGESIZE) - 1)) & ~(sysconf(_SC_PAGESIZE) - 1);
					if (shm_stat.st_size == 0)
					{
						shm_init = SVG_FALSE;
						if ( ftruncate( fd, p_shm_size[0] ) != 0 )
						{
						    SVG_FNT_E("SHM_SIZE_FAILED IN CREATE_SHMEMORY");
							mmap_required   = SVG_FALSE;
						}
						else
						{
							mmap_required   = SVG_TRUE;
						}
					}
					else
					{
						shm_init = SVG_TRUE;
						mmap_required   = SVG_TRUE;
					}
				}
			}

			if(mmap_required   == SVG_TRUE)
			{
				/* Map these sengments in the process memory space */
				mmap_struct = mmap( NULL, p_shm_size[0], PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0 );

				if ( mmap_struct == MAP_FAILED )
				{
				    SVG_FNT_E("SHM_MMAP_FAILED IN CREATE_SHMEMORY");
					ret = SVG_FALSE;
				}
				else
				{
					if(0 == strcmp(shm_name,"/svg_flashfonts"))
					{
						/* upadating flash font struct, to shared memory and from shared memory */
						gp_svgfont_flashshared = (shm_struct_flash *)mmap_struct;
						if (shm_init == SVG_FALSE)
						{
							for(i = 0; i< p_font_max[0] ;i++)
							{
								memcpy( &gp_svgfont_flashshared[i].flash_data ,
											&p_font_data[0][i] , (sizeof(SVGFont)) );

							}
							gp_svgfont_flashshared->num_flash_files = p_font_max[0];
							shm_init = SVG_TRUE;

						}

						if (shm_init == SVG_TRUE)
						{
							p_font_max[0]=gp_svgfont_flashshared->num_flash_files ;
							if(*p_font_data == NULL)
							{
								*p_font_data = (SVGFont*)GRL_malloc_1D_resource(sizeof(SVGFont) * p_font_max[0]);
							}
							for(i = 0; i< p_font_max[0] ;i++)
							{
								strcpy((char *)&gp_svgfont_flashshared[i].flash_data.filename ,
												(char *)(uintptr_t)p_file_list[i] );
								memcpy(&p_font_data[0][i],
									&gp_svgfont_flashshared[i].flash_data,sizeof(SVGFont));

							}
						}
					}
					else
					{
						/* upadating file font struct, to shared memory and from shared memory */
						gp_svgfont_fontshared = (shm_struct_font *)mmap_struct;

						if (shm_init == SVG_FALSE)
						{
							for(i = 0; i< p_font_max[0] ;i++)
							{
								memcpy( &gp_svgfont_fontshared[i].font_data ,
											&p_font_data[0][i] ,(sizeof(SVGFont)) );
							}
							gp_svgfont_fontshared->num_font_files = p_font_max[0];
							shm_init = SVG_TRUE;

						}

						if (shm_init == SVG_TRUE)
						{
							p_font_max[0] = gp_svgfont_fontshared->num_font_files;
							if(*p_font_data == NULL)
							{
								*p_font_data = (SVGFont*)GRL_malloc_1D_resource(sizeof(SVGFont) * p_font_max[0]);
							}
							for(i = 0; i< p_font_max[0] ;i++)
							{
								memcpy(&p_font_data[0][i],&gp_svgfont_fontshared[i].font_data,sizeof(SVGFont));
							}

						}
					}
				}
				close(fd);

			}
			grl_error = sem_post(GRL_FONT_LOCK_ID);
			if( 0 != grl_error )
			{
			    SVG_FNT_E("SEM_POST_FAILED IN CREATE_SHMEMORY");
				ret = SVG_FALSE;
			}
		}
		else
		{
		    SVG_FNT_E("SEM_WAIT_FAILED IN CREATE_SHMEMORY");
			ret = SVG_FALSE;
		}
	}
	return ret;
}



/**
 * @fn local_fill_struct
 *
 * files out the flash or file font struct
 *
 * @param SVGInt8                    : file font list with paths full path
 * @param SVGUint32                   : numbers of files
 * @param grl_module_info       : module informations to fill out
 * @param SVGBoolean            : identifies if its flash or file font
 *
 * @return SVGBoolean           : TRUE if successfull
 *
 */
static SVGBoolean  local_fill_struct(SVGUint32 *p_file_list ,
                              SVGUint32 num_file_names,
                              grl_module_info* p_module_info,
                              SVGBoolean is_flash_font)
{
    SVGBoolean          ret             = SVG_TRUE;
    SVGFont            *p_font_data     = NULL;
    SVGUint32           font_max        = 0;
    SVGUint32			i				= 0;
    SVGUint32           shm_size 		= 0;
    SVGChar             shm_name[256]   = {0};

	/**
	 * fill out every font struct and
	 * delete the filename
	 */
    font_max = 0;     
/* PRQA: 306: pointers are checked */
/*PRQA S 306 L1 */

    if(is_flash_font == FLASH_FONT)
    {
        strcpy(&shm_name[0],"/svg_flashfonts");

    }
    else
    {
        strcpy(&shm_name[0],"/svg_filefonts");
    }

	/* checking for existence of shared memory or parsed font file */

	ret = check_shmemory(shm_name ,&shm_size, p_file_list ,
							&p_font_data, &font_max ,num_file_names);

    /* when there is no shared memory and parsed file then only
     *  ret is equal to SVG_FALSE
     */
    if(ret == SVG_FALSE)
    {
        p_module_info->p_module->p_grl_fp_table->
            fill_font_structs( p_file_list, num_file_names, &p_font_data, &font_max );

        shm_size = (sizeof(SVGFont) * font_max) + sizeof(SVGUint32);
        ret = SVG_TRUE ;
    }

    ret = create_shmemory(shm_name , &shm_size ,&p_font_data , &font_max ,p_file_list);

   /* if there is any error in updating from shared memory */
    if(ret == SVG_FALSE)
   {

        p_module_info->p_module->p_grl_fp_table->
                fill_font_structs( p_file_list, num_file_names, &p_font_data, &font_max );

   }


/*PRQA L L1 */
    if (NULL != p_font_data)
    {
        if (is_flash_font == FLASH_FONT)
        {
            /* set local data for flash fonts */
            p_module_info->p_svgFlashFonts  = p_font_data;
            p_module_info->svgFlashFontsMax = font_max;
        }
        else
        {
            /* set local data for file fonts */
            p_module_info->p_svgFileFonts  = p_font_data;
            p_module_info->svgFileFontsMax = font_max;
        }
    }
    else
    {
        ret = SVG_FALSE;
    }
    /* destroy file name space */
    for (i= 0; i < num_file_names; i++)
    {
/* PRQA: 306: pointers are checked */
/*PRQA S 306 L1 */
        GRL_free_1D_resource((void*)(uintptr_t)p_file_list[i]);
/*PRQA L  L1 */
    }

    return ret;
}


/**
 * @fn GRL_num_of_flash_font
 *
 * get the number of flash fonts in the module
 *
 * @param SVGInt8                    : Suffix list of the module with point
 *                                and spaced by an tab: e.g. ".ttf\t.bin\t"
 *
 * @return SVGUint32                  : Number of flash fonts
 *
 */
static SVGUint32 local_num_of_flash_font( const SVGInt8 *p_suffix_list )
{
    SVGUint32     i                   = 0;
    SVGChar* p_cmpbuff           = NULL;
    SVGUint32     numFonts            = 0;

    for ( i = 0; i < GRL_FLASH_FONT_COUNT; i++ )
    {
/* PRQA: 311, 158: pointers are checked */
/*PRQA S 311  L1 */
/* PRQA: Lint Message 158: pointers are checked */
/*lint -save -e158 */
        p_cmpbuff = strstr( (const SVGChar*)p_suffix_list, (const SVGChar*)flash_fonts[i].suffix );
/*PRQA L L1 */
/*lint -restore */
        if (p_cmpbuff != NULL)
        {
            numFonts++;
        }
    }

    return numFonts;
}


/**
 * @fn GRL_is_valid_suffix
 *
 * control if the flash font has a valid suffix
 *
 * @param grl_flash_font        : The controlled flash font
 * @param unsigned char         : Suffix list with point and spaced
 *                                by an tab: e.g. ".ttf\t.bin\t"
 *
 * @return TRUE or FALSE
 *
 */
static SVGBoolean local_is_valid_suffix(grl_flash_font flash_font,
                                 const SVGInt8* p_suffix_list )
{
    SVGBoolean  ret         = SVG_FALSE;
    SVGChar     *p_cmpbuff  = NULL;
/* PRQA: 288: pointers are checked */
/*PRQA S 311 L1 */
/* PRQA: Lint Message 158: pointers are checked */
/*lint -save -e158 */
    p_cmpbuff = strstr( (const SVGChar*)p_suffix_list, (const SVGChar*)flash_font.suffix );
/*PRQA L  L1 */
/*lint -restore */
    if (p_cmpbuff != NULL)
    {
        ret = SVG_TRUE;
    }

    return ret;
}


/**
 * @fn GRL_get_flash_font_file
 *
 * get the flash font file name with address and length in the name
 *
 * @param grl_flash_font        : Pointer to the flash font
 * @param unsigned char         : Filename
 *
 * @return unsigned char        : Filename
 *
 */
static SVGChar *local_get_flash_font_file( const grl_flash_font* flash_font, SVGChar *name)
{
    SVGChar *cpy_str = NULL;

    name = strcpy(name, FF_ROOT);
    cpy_str = memcpy(&name[FF_ROOT_LEN], &flash_font->start_address, FF_INTEGER_LEN);
    cpy_str = memcpy(&cpy_str[FF_INTEGER_LEN], FF_SLASH, 1);
    cpy_str = memcpy(&cpy_str[1], &flash_font->length, FF_INTEGER_LEN);
    cpy_str = memcpy(&cpy_str[FF_INTEGER_LEN], FF_SLASH,1);
    cpy_str = memcpy(&cpy_str[1], flash_font->suffix, FF_INTEGER_LEN);

    return (name);
}

static SVGError local_build_file_font_list(SVGUint32       *p_nameList,
                                    SVGUint32      *FileCount,
                                    const SVGInt8 *p_suffix_list )
{
    SVGError          err            = SVG_NO_ERROR;
	SVGInt32          a_length       = 0;
    SVGUint32         i              = 0;
    SVGUint32         svg_maxpath    = 0;
    SVGChar           svgpathname[256] = {0};
    SVGChar           devconf_buf[256] = {0};
    SVGBoolean        use_devconf    = SVG_FALSE;
    SVGChar           *tmp           = NULL;
    SVGUint32         listCnt        = 0;

    if (NULL != FileCount)
    {
        *FileCount = 0;
    }

    /* read out number of paths */
    a_length = GRL_get_int_config((SVGChar*)"SVGPATHCOUNT", (SVGInt32*)&svg_maxpath, 0);
    if( a_length > 0)
    {
        GRL_get_int_config((SVGChar*)"SVGPATHCOUNT", (SVGInt32*)&svg_maxpath, (SVGUint8)a_length);
        /* are the pathnames available? */
        a_length = GRL_get_string_config((SVGChar*)"SVGPATHNAMES", (SVGUint8*)&devconf_buf[0], 0);
        if((a_length > 0) && (256 > a_length))
        {
            /* read out the pathnames and use them */
            GRL_get_string_config((SVGChar*)"SVGPATHNAMES", (SVGUint8*)&devconf_buf[0], (SVGUint8)a_length);
            if(1 == svg_maxpath)
            {
                memcpy(&svgpathname[0],(SVGUint8*)&devconf_buf[0],(SVGUint32)a_length);
            }
            else
            {
                tmp = strtok((SVGChar*)&devconf_buf,",");
                if (NULL != tmp)
                {
                    strncpy(&svgpathname[0], tmp, 256);
                }
            }
            use_devconf = SVG_TRUE;
        }
        else
        {
            /* in case of any unavailable or invalid value 
               just use internal paths for parsing */
            svg_maxpath = SVG_MAXPATH;
        }
    }

    for ( i = 0; (i < svg_maxpath) && (err == SVG_NO_ERROR); i++ )
    {
		if(SVG_TRUE == use_devconf)
        {
            /* use the values from devconf */
            if(0 < i)
            {
                tmp = strtok(NULL,",");
                if ( NULL != tmp)
                {
                    strncpy(&svgpathname[0], tmp, 256);
                }
            }
        }
        else
        {
            /* use the internal defined paths for parsing */
            strncpy(&svgpathname[0], (const SVGChar*)&svgPathNames[i][0], 256);
        }

        err = grl_font_scan_dir(p_nameList,
                                FileCount,
                                (const SVGChar*)p_suffix_list ,
                                &svgpathname[0],
                                &listCnt );
/*PRQA L L3 */
    }
    return err;
}

/*PRQA L L1 */
/*PRQA L L2 */
